import videojs from 'video.js';
import {
  fullScreenIcon,
  exitFullScreenIcon,
} from "../../../assets/icons/index.js";
import '../../../styles/fullscreen-button.css';

const Button = videojs.getComponent('Button');

class FullscreenButton extends Button {
  constructor(player, options) {
    super(player, options);
    this.controlText('Fullscreen');
    this.hideTimeout = null;
    this.isFullscreen = false;
    this.isMouseOverControls = false;
    this.controlBar = options.controlBar || null;
    this.isTouchDevice = false;
    this.isTouching = false;
  }

  createEl() {
    const el = super.createEl('button', {
      className: 'videojs-drm-button videojs-drm-fullscreen-button'
    });
    
    el.innerHTML = fullScreenIcon;
    
    return el;
  }

  handleClick() {
    if (this.player().isFullscreen()) {
      this.player().exitFullscreen();
    } else {
      this.player().requestFullscreen();
    }
  }

  updateFullscreenIcon(isFullscreen) {
    this.el().innerHTML = isFullscreen ? exitFullScreenIcon : fullScreenIcon;
  }

  ready() {
    super.ready();
    
    // Listen to player events
    this.player().on('fullscreenchange', () => {
      this.isFullscreen = this.player().isFullscreen();
      this.updateFullscreenIcon(this.isFullscreen);
      this.handleFullscreenChange();
    });

    this.player().on('enterfullscreen', () => {
      this.isFullscreen = true;
      this.updateFullscreenIcon(true);
      this.handleFullscreenChange();
    });

    this.player().on('exitfullscreen', () => {
      this.isFullscreen = false;
      this.updateFullscreenIcon(false);
      this.handleFullscreenChange();
    });
  }

  handleFullscreenChange() {
    if (this.isFullscreen) {
      // In fullscreen, start auto-hide behavior
      this.setupAutoHideBehavior();
      this.resetHideTimeout();
    } else {
      // Not in fullscreen, always show controls
      this.showControls();
      this.clearHideTimeout();
      this.removeAutoHideBehavior();
    }
  }

  setupAutoHideBehavior() {
    if (!this.controlBar) return;

    const playerEl = this.player().el();
    
    // Detect touch device
    this.isTouchDevice = 'ontouchstart' in window || navigator.maxTouchPoints > 0;

    if (!this.isTouchDevice) {
      // Mouse events for desktop
      const handleMouseActivity = (event) => {
        this.showControls();
        this.resetHideTimeout();
      };

      const handleControlMouseEnter = () => {
        this.isMouseOverControls = true;
      };

      const handleControlMouseLeave = () => {
        this.isMouseOverControls = false;
      };

      // Add event listeners
      playerEl.addEventListener('click', handleMouseActivity);
      playerEl.addEventListener('mousemove', handleMouseActivity);
      this.controlBar.addEventListener('mouseenter', handleControlMouseEnter);
      this.controlBar.addEventListener('mouseleave', handleControlMouseLeave);

      this.handleMouseActivity = handleMouseActivity;
      this.handleControlMouseEnter = handleControlMouseEnter;
      this.handleControlMouseLeave = handleControlMouseLeave;
    } else {
      // Touch events for mobile
      const handleTouchStart = () => {
        this.isTouching = true;
        this.showControls();
      };

      const handleTouchMove = () => {
        this.showControls();
        this.resetHideTimeout();
      };

      const handleTouchEnd = () => {
        this.isTouching = false;
        this.resetHideTimeout();
      };

      playerEl.addEventListener('touchstart', handleTouchStart, { passive: true });
      playerEl.addEventListener('touchmove', handleTouchMove, { passive: true });
      playerEl.addEventListener('touchend', handleTouchEnd, { passive: true });
      playerEl.addEventListener('touchcancel', handleTouchEnd, { passive: true });

      this.handleTouchStart = handleTouchStart;
      this.handleTouchMove = handleTouchMove;
      this.handleTouchEnd = handleTouchEnd;
    }
  }

  removeAutoHideBehavior() {
    if (!this.controlBar) return;

    const playerEl = this.player().el();

    // Remove event listeners
    if (this.handleMouseActivity) {
      playerEl.removeEventListener('click', this.handleMouseActivity);
      playerEl.removeEventListener('mousemove', this.handleMouseActivity);
    }
    if (this.handleControlMouseEnter) {
      this.controlBar.removeEventListener('mouseenter', this.handleControlMouseEnter);
    }
    if (this.handleControlMouseLeave) {
      this.controlBar.removeEventListener('mouseleave', this.handleControlMouseLeave);
    }
    if (this.handleTouchStart) {
      playerEl.removeEventListener('touchstart', this.handleTouchStart);
    }
    if (this.handleTouchMove) {
      playerEl.removeEventListener('touchmove', this.handleTouchMove);
    }
    if (this.handleTouchEnd) {
      playerEl.removeEventListener('touchend', this.handleTouchEnd);
      playerEl.removeEventListener('touchcancel', this.handleTouchEnd);
    }

    // Clear references
    this.handleMouseActivity = null;
    this.handleControlMouseEnter = null;
    this.handleControlMouseLeave = null;
    this.handleTouchStart = null;
    this.handleTouchMove = null;
    this.handleTouchEnd = null;
  }

  showControls() {
    if (this.controlBar) {
      this.controlBar.style.opacity = '1';
      this.controlBar.style.visibility = 'visible';
    }
  }

  hideControls() {
    if (this.controlBar && this.isFullscreen && !this.isMouseOverControls && !this.isTouching) {
      this.controlBar.style.opacity = '0';
      this.controlBar.style.visibility = 'hidden';
    }
  }

  resetHideTimeout() {
    this.clearHideTimeout();
    if (this.isFullscreen) {
      // Different timeout for touch vs mouse devices
      const timeout = this.isTouchDevice ? 3000 : 4000; // 3s for touch, 4s for mouse
      this.hideTimeout = setTimeout(() => {
        this.hideControls();
      }, timeout);
    }
  }

  clearHideTimeout() {
    if (this.hideTimeout) {
      clearTimeout(this.hideTimeout);
      this.hideTimeout = null;
    }
  }

  dispose() {
    // Clean up event listeners and timeouts
    this.clearHideTimeout();
    this.removeAutoHideBehavior();
    super.dispose();
  }
}

// Register the component
videojs.registerComponent('FullscreenButton', FullscreenButton);

export default FullscreenButton;
