import videojs from 'video.js';
import { 
  PlayButton, 
  VolumeButton, 
  VolumeSlider, 
  TimeDisplay, 
  CCButton,
  SettingsButton,
  FullscreenButton,
  ProgressBar
} from './control-bar/components';
import '../styles/custom-control-bar.css';

const Component = videojs.getComponent('Component');

class CustomControlBar extends Component {
  constructor(player, options) {
    super(player, options);
    this.isAudioOnly = options.isAudioOnly;

    console.log('CustomControlBar ready, setting up controls...');
    this.setupControls();
  }

  createEl() {
    const el = super.createEl('div', {
      className: 'custom-video-control-bar'
    });

    // Create control bar HTML structure
    el.innerHTML = `
      <div class="custom-progress-container"></div>
      <div class="custom-controls-container">
        <div class="custom-controls-left">
          <div class="custom-play-button-container"></div>
          <div class="custom-volume-container"></div>
          <div class="custom-time-display-container"></div>
        </div>
        <div class="custom-controls-right">
          <div class="custom-cc-button-container"></div>
          <div class="custom-settings-button-container"></div>
          <div class="custom-fullscreen-button-container"></div>
        </div>
      </div>
    `;

    return el;
  }

  ready() {
    super.ready();
    console.log('CustomControlBar ready!');
  }

  setupControls() {
    // Detect touch device for volume slider hover
    const isTouchDevice = 'ontouchstart' in window || navigator.maxTouchPoints > 0;

    // Create and add Progress Bar
    const progressBar = new ProgressBar(this.player(), {});
    const progressContainer = this.el().querySelector('.custom-progress-container');
    progressContainer.appendChild(progressBar.el());

    // Create and add Play Button
    const playButton = new PlayButton(this.player(), {});
    const playContainer = this.el().querySelector('.custom-play-button-container');
    playContainer.appendChild(playButton.el());

    // Create and add Volume Button
    const volumeContainer = this.el().querySelector('.custom-volume-container');
    const volumeButton = new VolumeButton(this.player(), {});
    volumeContainer.appendChild(volumeButton.el());
    if (!isTouchDevice) {
      const volumeSlider = new VolumeSlider(this.player(), {});
      volumeContainer.appendChild(volumeSlider.el());
      this.setupVolumeSliderHover(volumeContainer, volumeSlider.el());
    }

    if (!this.isAudioOnly) {
      // Create and add Time Display
      const timeDisplay = new TimeDisplay(this.player(), {});
      const timeContainer = this.el().querySelector('.custom-time-display-container');
      timeContainer.appendChild(timeDisplay.el());

      // Create and add CC Button
      const ccButton = new CCButton(this.player(), {});
      const ccContainer = this.el().querySelector('.custom-cc-button-container');
      ccContainer.appendChild(ccButton.el());

      // Create and add Settings Button
      const settingsButton = new SettingsButton(this.player(), {});
      const settingsContainer = this.el().querySelector('.custom-settings-button-container');
      settingsContainer.appendChild(settingsButton.el());

      // Create and add Fullscreen Button
      const fullscreenButton = new FullscreenButton(this.player(), {
        controlBar: this.el()
      });
      const fullscreenContainer = this.el().querySelector('.custom-fullscreen-button-container');
      fullscreenContainer.appendChild(fullscreenButton.el());
    }
  }


  // TODO: Relocate logic
  setupVolumeSliderHover(vContainer, vSlider) {
    if (!vContainer || !vSlider) return;

    // Hide volume slider when mouse leaves both button and slider
    const hideSlider = () => {
      vSlider.style.opacity = "0";
      vSlider.style.width = "0";
      vSlider.style.paddingRight = "0";
    };

    // Initially hide the volume slider
    hideSlider();

    // Show volume slider on hover
    vContainer.addEventListener("mouseenter", () => {
      this.clearVolumeTimer();
      vSlider.style.opacity = "1";
      vSlider.style.width = null;
      vSlider.style.paddingRight = "4px";
    });

    vContainer.addEventListener("mouseleave", (e) => {
      this.volumnTimer = setTimeout(() => {
        hideSlider();
      }, 200);
    });
  }

  clearVolumeTimer() {
    if (this.volumnTimer) {
      clearTimeout(this.volumnTimer);
      this.volumnTimer = null;
    }
  }

  dispose() {
    // TODO: Clear all event listeners
    this.clearVolumeTimer();
    super.dispose();
  }
}

// Register the component
videojs.registerComponent('CustomControlBar', CustomControlBar);

export default CustomControlBar;
