// Volume Control Component Class
import {
  volumeMutedIcon,
  volumeLowIcon,
  volumeHighIcon,
} from "../../../assets/icons/index.js";
import { createButton } from "../../../utils/buttonFactory.js";

const isTouchDevice = "ontouchstart" in window || navigator.maxTouchPoints > 0;

export class VolumeControlComponent {
  constructor(player) {
    this.player = player;

    this.container = null;
    this.volumeBtn = null;
    this.volumeSlider = null;
    this.volumeTimeout = null;

    this.init();
  }

  init() {
    this.createElements();
    this.setupEventListeners();
    this.updateVolumeIcon();
  }

  createElements() {
    this.container = document.createElement("div");
    this.container.className = "pp-volume-container";

    this.volumeBtn = createButton("volume", "Volume", () => {
      this.player.setMuted(!this.player.isMuted());
    });

    // Volume slider (hidden by default, show on hover)
    this.volumeSlider = document.createElement("input");
    this.volumeSlider.type = "range";
    this.volumeSlider.className = "pp-volume-slider pp-volume-slider-hidden";
    this.volumeSlider.min = "0";
    this.volumeSlider.max = "1";
    this.volumeSlider.step = "0.01";
    this.volumeSlider.value = this.player.getVolume();

    // Initialize volume slider background gradient
    const initialVolumePercent = (this.player.getVolume() / 1) * 100;
    this.volumeSlider.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${initialVolumePercent}%, rgba(255, 255, 255, 0.3) ${initialVolumePercent}%)`;

    this.container.appendChild(this.volumeBtn);
    if (!isTouchDevice) {
      this.container.appendChild(this.volumeSlider);
    }
  }

  updateVolumeIcon() {
    const effectiveVolume = this.player.isMuted() ? 0 : this.player.getVolume();
    this.volumeBtn.innerHTML =
      effectiveVolume === 0
        ? volumeMutedIcon
        : effectiveVolume < 0.5
        ? volumeLowIcon
        : volumeHighIcon;
  }

  handleVolumeChange() {
    this.updateVolumeIcon();

    this.volumeSlider.value = this.player.getVolume();
    // Update volume slider background gradient
    const volumePercent = (this.player.getVolume() / 1) * 100;
    this.volumeSlider.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${volumePercent}%, rgba(255, 255, 255, 0.3) ${volumePercent}%)`;
  }

  setupEventListeners() {
    // Volume slider styling
    this.volumeSlider.addEventListener("input", (e) => {
      this.player.setVolume(parseFloat(e.target.value));
    });

    this.player.on("volumechange", this.handleVolumeChange.bind(this));

    // Show/hide volume slider on hover
    this.container.addEventListener("mouseenter", () => {
      clearTimeout(this.volumeTimeout);
      this.volumeSlider.classList.add("pp-volume-slider-visible");
      this.volumeSlider.classList.remove("pp-volume-slider-hidden");
    });

    this.container.addEventListener("mouseleave", () => {
      clearTimeout(this.volumeTimeout);
      this.volumeTimeout = setTimeout(() => {
        this.volumeSlider.classList.remove("pp-volume-slider-visible");
        this.volumeSlider.classList.add("pp-volume-slider-hidden");
      }, 300);
    });
  }

  cleanup() {
    if (this.volumeTimeout) {
      clearTimeout(this.volumeTimeout);
    }
    // TODO: Add cleanup for event listeners
  }
}
