import {
  fullScreenIcon,
  exitFullScreenIcon,
} from "../../../assets/icons/index.js";
import { createButton } from "../../../utils/buttonFactory.js";
import { Fullscreen } from "./fullscreen-engine.js";

const iOS = /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream;
const isTouchDevice = "ontouchstart" in window || navigator.maxTouchPoints > 0;

export class FullscreenComponent {
  constructor(player, video, playerContainer, controlsContainer) {
    this.player = player;
    this.video = video;
    this.playerContainer = playerContainer;
    this.controlsContainer = controlsContainer;

    this.fullscreenEngine = new Fullscreen(
      iOS ? video : playerContainer,
      iOS ? video : document,
      this.handleFullscreenChange.bind(this)
    );

    this.init();
  }

  init() {
    this.createElements();
  }

  createElements() {
    this.container = document.createElement("div");
    this.container.className = "pp-fullscreen-container";

    this.fullscreenBtn = createButton("fullscreen", "Fullscreen", () => {
      if (this.fullscreenEngine.fullscreenEnabled()) {
        this.fullscreenEngine.exitFullscreen();
      } else {
        this.fullscreenEngine.requestFullscreen();
      }
    });

    this.container.appendChild(this.fullscreenBtn);
  }

  // Function to update fullscreen icon
  updateFullscreenIcon() {
    this.fullscreenBtn.innerHTML = this.fullscreenEngine.fullscreenEnabled()
      ? exitFullScreenIcon
      : fullScreenIcon;
    this.fullscreenBtn.title = this.fullscreenEngine.fullscreenEnabled()
      ? "Exit Fullscreen"
      : "Fullscreen";
  }

  showControls() {
    if (!this.controlsContainer) return;
    this.controlsContainer.style.opacity = "1";
    this.controlsContainer.style.visibility = "visible";
  }

  hideControls() {
    if (
      this.controlsContainer &&
      this.fullscreenEngine.fullscreenEnabled() &&
      !this.isMouseOverControls &&
      !this.isTouching
    ) {
      this.controlsContainer.style.opacity = "0";
      this.controlsContainer.style.visibility = "hidden";
    }
  }

  resetHideTimeout() {
    this.clearHideTimeout();
    if (!this.fullscreenEngine.fullscreenEnabled()) return;

    // Different timeout for touch vs mouse devices
    const timeout = isTouchDevice ? 3000 : 4000; // 3s for touch, 4s for mouse
    this.hideTimeout = setTimeout(this.hideControls.bind(this), timeout);
  }

  clearHideTimeout() {
    if (this.hideTimeout) {
      clearTimeout(this.hideTimeout);
      this.hideTimeout = null;
    }
  }

  setupAutoHideBehavior() {
    if (!this.controlsContainer) return;

    if (!isTouchDevice) {
      // Mouse events for desktop
      const handleMouseActivity = (event) => {
        this.showControls();
        this.resetHideTimeout();
      };

      const handleControlMouseEnter = () => {
        this.isMouseOverControls = true;
      };

      const handleControlMouseLeave = () => {
        this.isMouseOverControls = false;
      };

      // Add event listeners
      this.playerContainer.addEventListener("click", handleMouseActivity);
      this.playerContainer.addEventListener("mousemove", handleMouseActivity);
      this.controlsContainer.addEventListener(
        "mouseenter",
        handleControlMouseEnter
      );
      this.controlsContainer.addEventListener(
        "mouseleave",
        handleControlMouseLeave
      );

      this.handleMouseActivity = handleMouseActivity;
      this.handleControlMouseEnter = handleControlMouseEnter;
      this.handleControlMouseLeave = handleControlMouseLeave;
    } else {
      // Touch events for mobile
      const handleTouchStart = () => {
        this.isTouching = true;
        this.showControls();
      };

      // const handleTouchMove = () => {
        // this.showControls();
        // this.resetHideTimeout();
      // };

      const handleTouchEnd = () => {
        this.isTouching = false;
        this.resetHideTimeout();
      };

      this.playerContainer.addEventListener("touchstart", handleTouchStart, {
        passive: true,
      });
      // this.playerContainer.addEventListener("touchmove", handleTouchMove, {
      //   passive: true,
      // });
      this.playerContainer.addEventListener("touchend", handleTouchEnd, {
        passive: true,
      });
      this.playerContainer.addEventListener("touchcancel", handleTouchEnd, {
        passive: true,
      });

      this.handleTouchStart = handleTouchStart;
      // this.handleTouchMove = handleTouchMove;
      this.handleTouchEnd = handleTouchEnd;
    }
  }

  removeAutoHideBehavior() {
    if (!this.controlsContainer) return;

    // Remove event listeners
    if (this.handleMouseActivity) {
      this.playerContainer.removeEventListener(
        "click",
        this.handleMouseActivity
      );
      this.playerContainer.removeEventListener(
        "mousemove",
        this.handleMouseActivity
      );
    }
    if (this.handleControlMouseEnter) {
      this.controlsContainer.removeEventListener(
        "mouseenter",
        this.handleControlMouseEnter
      );
    }
    if (this.handleControlMouseLeave) {
      this.controlsContainer.removeEventListener(
        "mouseleave",
        this.handleControlMouseLeave
      );
    }
    if (this.handleTouchStart) {
      this.playerContainer.removeEventListener(
        "touchstart",
        this.handleTouchStart
      );
    }
    // if (this.handleTouchMove) {
    //   this.playerContainer.removeEventListener(
    //     "touchmove",
    //     this.handleTouchMove
    //   );
    // }
    if (this.handleTouchEnd) {
      this.playerContainer.removeEventListener("touchend", this.handleTouchEnd);
      this.playerContainer.removeEventListener(
        "touchcancel",
        this.handleTouchEnd
      );
    }

    // Clear references
    this.handleMouseActivity = null;
    this.handleControlMouseEnter = null;
    this.handleControlMouseLeave = null;
    this.handleTouchStart = null;
    // this.handleTouchMove = null;
    this.handleTouchEnd = null;
  }

  handleFullscreenChange() {
    this.showControls();
    this.updateFullscreenIcon();
    if (this.fullscreenEngine.fullscreenEnabled()) {
      this.setupAutoHideBehavior();
      this.resetHideTimeout();
    } else {
      this.clearHideTimeout();
      this.removeAutoHideBehavior();
    }
  }

  cleanup() {
    this.fullscreenEngine.destroy();
    this.clearHideTimeout();
    this.removeAutoHideBehavior();
  }
}
