/**
 * This class manages data storage in the browser memory.
 *
 */
export default class BrowserStorage {
    private prefix;
    private disableCookies;
    private forceCookies;
    private disabled;
    /**
     *
     * @constructs BrowserStorage
     *
     * @param {string} [prefix] Optional. Sets the prefix for saved objects in storages or cookies. 'npaw' by default.
     * @param {boolean} [disableCookies] Optional. Set to true to disable cookies fallback. True by default.
     * @param {boolean} [forceCookies] Optional. Set to true to disable cookies fallback. True by default.
     * @param {boolean} [disabled] Optional. Set to true to disable cookies fallback. True by default.
     */
    constructor(disableCookies: boolean, forceCookies: boolean, disabled: boolean);
    /**
     *
     * @param disableCookies
     * @param forceCookies
     * @param disable
     */
    updateStorageOptions(disableCookies: boolean, forceCookies: boolean, disable: boolean): void;
    /**
     * Returns if storages are available or not
     *
     */
    isEnabled(): boolean;
    /**
     * Saves in localStorage or equivalent
     *
     * @param {string} key Key of the value. Prefix will be appended.
     * @param {string} value Value.
     */
    setLocal(key: string, value: string): void;
    /**
     * Reads from localStorage or equivalent
     *
     * @param {string} key Key of the value. prefix will be appended.
     */
    getLocal(key: string): string | undefined;
    /**
     * Removes from localStorage or equivalent
     *
     * @param {string} key Key of the value. prefix will be appended.
     */
    removeLocal(key: string): string | undefined;
    private _localGetRemove;
    /**
     * Saves in sessionStorage or equivalent
     *
     * @param {string} key Key of the value. prefix will be appended.
     * @param {string} value Value.
     */
    setSession(key: string, value: string): void;
    /**
     * Reads from sessionStorage or equivalent
     *
     * @param {string} key Key of the value. prefix will be appended.
     */
    getSession(key: string): string | undefined;
    /**
     * Removes from sessionStorage or equivalent
     *
     * @param {string} key Key of the value. prefix will be appended.
     */
    removeSession(key: string): string | undefined;
    private _sessionGetRemove;
    /**
     * Calls getSession and getLocal for the same key
     * @param {string} key Key of the value. prefix will be appended.
     */
    getStorages(key: string): string | undefined;
    /**
     * Calls getSession and getLocal with the same key and value
     * @param {string} key Key of the value. prefix will be appended.
     * @param {string} value Value.
     */
    setStorages(key: string, value: string): void;
    /**
     * Calls getSession and getLocal for the same key
     * @param {string} key Key of the value. prefix will be appended.
     */
    removeStorages(key: any): void;
    /**
     * Sets a cookie value
     *
     * @param {string} cname Key of the value.
     * @param {Object} cvalue Value.
     */
    private _setCookie;
    /**
     * Gets a cookie value
     *
     * @param {string} cname Key of the value.
     */
    private _getCookie;
    /**
     * Removes a cookie
     *
     * @param {string} cname Key of the value.
     */
    private _removeCookie;
}
