export declare enum LogLevel {
    /** No console outputs */
    SILENT = 6,
    /** Console will show errors */
    ERROR = 5,
    /** Console will show warnings */
    WARNING = 4,
    /** Console will show notices (ie: life-cycle logs) */
    NOTICE = 3,
    /** Console will show debug messages (ie: player events) */
    DEBUG = 2,
    /** Console will show verbose messages (ie: Http Requests) */
    VERBOSE = 1
}
/**
 * Static Log class for NPAW Plugin
 */
export default class Log {
    /**
     * Only logs of this imporance or higher will be shown.
     * @default LogLevels.ERROR
     */
    private static logLevel;
    /**
     * If true, console logs will always be outputed without colors (for debbugin in devices).
     * @default false
     */
    private static plainLogs;
    /**
     * Returns a console message
     *
     * @private
     * @param tag Tag used to identify component
     * @param msg Message string, error object or array of messages.
     * @param level Defines the level of the error sent.
     * Only errors with higher or equal level than Log.logLevel will be displayed.
     * @param color Color of the header
     */
    private static report;
    /**
     * Returns the current time in format hh:mm:ss.mmm (with trailing 0s)
     * @return Current time.
     */
    private static _getCurrentTime;
    /**
     * Returns a console message without style
     *
     * @param msg Message string, object or array of messages.
     * @param prefix Prefix of the message.
     */
    private static _plainReport;
    /**
     * Defines the level of the NPAW Plugin logging
     *
     * @param logLevel Logging level
     */
    static setLogLevel(logLevel: number): void;
    /**
     * Gets the current log level of the NPAW Plugin
     *
     * @return The current log level.
     */
    static getLogLevel(): number;
    /**
     * Sends an error (level 1) console log.
     * Supports unlimited arguments: ("this", "is", "a", "message")
     */
    static error(tag: string, ...msg: any): void;
    /**
     * Sends a warning (level 2) console log.
     * Supports unlimited arguments: ("this", "is", "a", "message")
     */
    static warn(tag: string, ...msg: any): void;
    /**
     * Sends a notice (level 3) console log.
     * Supports unlimited arguments: ("this", "is", "a", "message")
     */
    static notice(tag: string, ...msg: any): void;
    /**
     * Sends a debug message (level 4) to console.
     * Supports unlimited arguments: ("this", "is", "a", "message")
     */
    static debug(tag: string, ...msg: any): void;
    /**
     * Sends a verbose message (level 5) to console.
     * Supports unlimited arguments: ("this", "is", "a", "message")
     */
    static verbose(tag: string, ...msg: any): void;
    /**
     * This function is automatically executed at npawlib's init.
     * Will search inside window.location.search for attribute 'npaw-debug=X'.
     * X can have one of these values, that will modify LogLevel.
     * 6: SILENT,
     * 5: ERROR,
     * 4: WARNING,
     * 3: NOTICE,
     * 2: DEBUG,
     * 1: VERBOSE
     *
     * If npaw-console=plain is present, plainLogs will be set to true.
     */
    static loadLevelFromUrl(): void;
    private static _parseLevelFromUrl;
}
