/// <reference types="node" />
import Emitter from '../Utils/Emitter';
import Resource from '../Resource/Resource';
import Rendition from '../Resource/Rendition';
/**
 * @class
 * @description Video segment class
 * It contains the url of the request, with host that can be modified to request it to a different cdn.
 * Also has unique id for every segment, and when requested, the response data and the size of it.
 * Additionally, when requested to a CDN, it stores the bandwidth calculated on the request
 * and when was it used on cache for last time.
 * @exports VideoSegment
 * @extends Emitter
 */
export declare class VideoSegment extends Emitter {
    id: string;
    p2pId: string;
    pathName: string;
    host: string;
    origRes: string;
    url: string;
    finalUrl: string;
    cdnName: string;
    cdnProvider: string;
    route: string;
    params: string;
    headers?: {
        [key: string]: string;
    };
    data: ArrayBuffer;
    size: number;
    cachedSize?: number;
    bandwidth: number;
    startTime: number;
    endTime: number;
    downloadTimeMs: number;
    retries: number;
    failed: boolean;
    timeout: boolean;
    useP2P: boolean;
    triedP2P: boolean;
    triedCdns: string[];
    arrayBuffer: boolean;
    progressDownloadedBytes: number;
    progressDownloadedBytesLastUpdate: number;
    instantBandwidth: number;
    duration: number;
    initialEstimatedTime: number;
    lastEstimatedTime: number;
    isManifest: boolean;
    isMasterPlaylist: boolean;
    isMedia: boolean;
    _response: any;
    private _resource;
    rendition: Rendition | undefined;
    download: boolean;
    realBodySize: number;
    readyForP2P: boolean;
    announcedP2P: boolean;
    activeP2PSendProcesses: number;
    retryBecauseNewHeaders: number;
    forceHttp: boolean;
    peer: any;
    createdAt: number;
    failedOriginalCount: number;
    requestReadTimeout?: NodeJS.Timeout;
    statusCode?: number;
    compressedBytes?: number;
    uncompressedBytes?: number;
    segmentDuration?: number;
    timestamp?: number;
    /**
     * Constructs VideoSegment.
     * @param {string} id id code of the segment.
     * @param {string} url initial full url of the video segment.
     * @param {Object} headers optional object with headers for the requests.
     * @param {number} retries number of retries before discarding the request of the segment, 3 by default.
     * @param {boolean} p2p boolean that indicates if the following request has to be done via p2p mechanism.
     * @param {boolean} arrayBuffer set as true if is expected to get the response in ArrayBuffer format.
     */
    constructor(pathName: string, url: string, resource: Resource, headers?: {
        [key: string]: string;
    }, retries?: number, p2p?: boolean, arrayBuffer?: boolean);
    updateP2PId(): void;
    getNewUrl(signManifest: boolean): string;
    addParametersToUrl(url: string | undefined): string | undefined;
    /**
     * Changes the target URL.
     * @param {string} newUrl New URL of the selected cdn we want to use.
     * @param {string} cdnName name of the new target CDN.
     * @public
     */
    changeCDN(newUrl: string, cdnName: string, cdnProvider: string): void;
    /**
     * Requests this video segment with the url it has (host, route, params).
     * Throws success event when done.
     * @param {boolean} secure Optional, if should be forced to use https.
     * @public
     */
    /**
     * Function to be called to set the data to this video segment externally.
     * For example when obtained the data by using P2P.
     * @param {ArrayBuffer} data The data object.
     * @public
     */
    setContent(data: ArrayBuffer): void;
    /**
     * Function to be called when an attempt to get the data externally, like using P2P, failed.
     * @public
     */
    noContentObtained(): void;
    private getPerfLoadTime;
    onProgress(xhr: XMLHttpRequest, event: ProgressEvent): void;
    onSuccess(response: any, contentLength: string, allowSpecialDelimitersInUrl?: boolean): boolean;
}
