import { VideoSegment } from '../Storage/VideoSegment';
import Peer from './Peer';
import BalancerOptions from '../Utils/Options';
import SegmentStorage from '../Storage/SegmentStorage';
import Loader from './Loader';
import Cdn from './Cdn';
declare global {
    interface Window {
        P2PManager: any | undefined;
    }
}
/**
 * @type {peer}
 * Peer defined by bittorrent-tracker.
 */
type peer = {
    id: string;
    on: (string: any, callback: any) => null;
    off: (string: any, callback: any) => null;
    write: (string: any) => null;
    destroy: () => null;
};
/**
 * @class
 * @description P2P manager class, that creates the P2P client and manages the peers and request segments to them.
 * @exports P2PLoader
 */
export default class P2PLoader {
    upload: boolean;
    private _isEnabled;
    private _packageFound;
    private _maxConcurrent;
    private _storage;
    private _accountCode;
    private _options;
    private _maxConnectedPeers;
    /** Peers managing */
    private _p2pManager?;
    private _peers;
    private _bannedPeers;
    private _candidates;
    private _activeDownloads;
    private _segmentsRequested;
    private _activeDownloadIds;
    private _monitoringStarted;
    private _performReset;
    /** Statistic params */
    private _downloadedPeers;
    private _uplodadedPeers;
    private _maxBandwidth;
    private _minBandwidth;
    private _timeoutDiscardedBytes;
    private _failedRequests;
    private _timeoutRequests;
    private _segmentAbsentRequests;
    private _errorRequests;
    private _downloadMillis;
    private _byteDownloadCount;
    private _chunkDownloadCount;
    private _sumResponseBytes?;
    private _minResponseBytes?;
    private _maxResponseBytes?;
    private _samplesResponseBytes?;
    private _sumResponseTime?;
    private _minResponseTime?;
    private _maxResponseTime?;
    private _samplesResponseTime?;
    private _sumNetworkLatency?;
    private _minNetworkLatency?;
    private _maxNetworkLatency?;
    private _samplesNetworkLatency?;
    private _sumThroughput?;
    private _minThroughput?;
    private _maxThroughput?;
    private _samplesThroughput?;
    private _sumVideoBytes?;
    private _sumVideoTime?;
    private _totalPeerSet;
    private _activePeerSet;
    private _availablePeersMap;
    private _peersAvailable?;
    private _maxPeersAvailable?;
    private _minPeersAvailable?;
    private _peersUsed?;
    private _peersParallelUsed?;
    private _maxPeersParallelUsed?;
    private _minPeersParallelUsed?;
    private _peerDiscoveryTime?;
    private _peerConnectionTime?;
    private _updatePeersMetricsRefCounter;
    private _offerInterval;
    private _accumBw;
    private _downloadMillisVideo;
    private _downloadedBytesVideo;
    private _downloadedChunksVideo;
    private _byteUploadDiscardedCount;
    private _chunkUploadDiscardedCount;
    private _chunkUploadCount;
    private _byteUploadCount;
    private _uploadRequests;
    private _uploadRequestsFailed;
    private _destroyed;
    private _cdnObject;
    private _cdnPingTimeBean;
    /**
     * Constructs P2PLoader.
     * @param accountCode
     * @param {BalancerOptions} options Options object.
     * @param {SegmentStorage} storage SegmentStore object.
     */
    constructor(accountCode: string, options: BalancerOptions, storage: SegmentStorage);
    isEnabled(): boolean;
    enable(): void;
    disable(): void;
    getStorage(): SegmentStorage;
    getUploadedUniqPeers(): number;
    getPeersAhead(): number;
    getPeersBehind(): number;
    getPeersNothing(): number;
    getId(): any;
    getCdnObject(): Cdn;
    resetP2PConnection(): void;
    sendJoin(): void;
    /**
     * Sets P2P settings from API response.
     * @param {balancerResponse} e API response.
     * @public
     */
    setSettings(e: balancerResponse): void;
    startMonitoring(): void;
    isMaxPeers(): boolean;
    getMissingPeers(): number;
    getMaxPeers(): number;
    getBannedPeers(): Map<string, boolean>;
    getUploadRequests(): number;
    getUploadRequestsFailed(): number;
    /**
     * @param {string} id Id of the segment.
     * @returns {VideoSegment|undefined} Segment object corresponding to the id, or undefined.
     * @public
     */
    getSegment(segment: VideoSegment): VideoSegment | undefined;
    /**
     * Return the list of peers that can serve the video segment
     *
     * @param segment Video segment object to be requested.
     */
    getPeersWithContent(segment: VideoSegment): Peer[];
    /**
     * Tries to check if the segment is available from any peer, and returns if it is or not.
     * @param {VideoSegment} seg Video segment object to be requested.
     * @returns {boolean} If the segment can be downloaded using P2P or not.
     * @public
     */
    request(segment: VideoSegment, loader?: Loader): boolean;
    addPeerResponse(response: responseStorageObject): void;
    private getLastResponses;
    addPeerLatency(peerLatency: number): void;
    /**
     * Callback for error event from P2P Client.
     * @param {Object} e Error event data.
     * @private
     * @static
     */
    private static _errorListener;
    /**
     * Callback for warning event from P2P Client.
     * @param {Object} e Warning event data.
     * @private
     * @static
     */
    private static _warningListener;
    /**
     * If p2p upload enabled, it stores the downloaded segment to share it with peers.
     * @param segment Segment to store.
     * @param data Data to store in the segment in memory.
     * @public
     */
    storeSegment(segment: VideoSegment): void;
    /**
     * Method to be called when we want to notify the peers that we have an updated map (list of video segments).
     * @private
     */
    private _sendMapToAllPeersV0;
    /**
     * Method to be called when we want to notify the peers I have a new segment available.
     * @private
     */
    private _sendNewSegmentToAllPeersV1;
    /**
     * Method to be called when we want to notify the peers that we have an updated map (list of video segments).
     * @private
     */
    private _sendMapToPeer;
    /**
     * Callback for new peer candidate event from P2P Client.
     * @param {callbackData} e Peer information to listen.
     * @private
     */
    addPeerCandidate(peer: peer, answer: any): void;
    /**
     * If the peer is connected and didnt exist adds it to peers list and removes the candidates.
     * If it existed before and was connected, it deletes it.
     * @param {Object} e Event object with event name and the new peer.
     * @private
     */
    peerConnect(peer: Peer): void;
    /**
     * Callback of close peer event. Will remove it from candidates and peers.
     * @param {Object} e Event object with event name and the peer to remove.
     * @private
     */
    peerCloseListener(data: Peer): void;
    /**
     * Callback of peer segment request event. Given a request if available it returns the chosen segment.
     * @param {Object} e Callback data from peer segment request event.
     * @private
     */
    peerSegmentUploadRequest(peer: Peer, segmentId: string, operationId: number): void;
    /**
     * Callback from peer segment loaded. It gets the data and adds it to the segment object,
     * then triggers the success event of it.
     * @param {Object} e Callback data from peer segment loaded event.
     * @private
     */
    peerSegmentLoaded(peer: Peer, id: string, time: number, data: ArrayBuffer): void;
    /**
     * Callback from peer segment loading progress. It gets the data and adds it to the segment object,
     * then triggers the success event of it.
     * @param {Object} e Callback data from peer segment loaded event.
     * @private
     */
    peerSegmentProgress(peer: Peer, id: string, time: number, data: ArrayBuffer, size: number): void;
    peerFailedSegmentTimeout(peer: Peer, id: string, size: number): void;
    peerFailedSegmentAbsent(peer: Peer, id: string): void;
    /**
     * Callback from peer segment uploaded. Counts the downloaded bytes and segments.
     * @param {Object} e callback data from peer segment uploaded event.
     * @private
     */
    peerSegmentUpload(size: number): void;
    getUploadedBytes(): number;
    getUploadedChunks(): number;
    /**
     * Callback from peer segment request cancelled. Counts the uploaded bytes that wont be used by the peer.
     * @param {Object} e callback data from peer segment upload failed event.
     * @private
     */
    peerCanceledSegmentUpload(size: number): void;
    /**
     * Destroys the client and the peers/candidates.
     * To be called when view is over or content switched.
     * @public
     */
    destroy(): void;
    /**
     * Returns the timestamp of the oldest request stored from all the peers.
     * @returns {number} Timestamp of the oldest request stored.
     * @public
     */
    getOldestRequestTS(): number;
    getPeers(): Map<string, Peer>;
    /**
     * Returns an object with the P2P data stats of the current content/view.
     * @returns {P2PLoaderStats} P2P info object.
     * @public
     */
    getStats(): P2PLoaderStats;
    resetOnPing(): void;
    onOffer(peerId: string, offerInterval: number): void;
    updatePeerMetrics(): void;
    onConnected(peerConnectionTime: number): void;
    onDiscovery(discoveryTime: number): void;
}
export {};
