/**
 * Shaka Player DRM Configuration
 */
import shaka from "shaka-player";

export async function setupShakaDRM(shakaPlayer, drmConfig = {}) {
  /**
   * Shaka Tech（Shaka Player）側の DRM 設定です。
   *
   * - Video.js + `videojs-contrib-eme` を使う場合は `src/drm-config.js` を通ります。
   * - `useShakaTech: true` の場合はこちらが使われ、Shaka の `drm.servers` を構成します。
   *
   * drmConfig は以下のような形を想定します：
   * {
   *   widevine: { url, licenseHeaders },
   *   playready: { url, licenseHeaders },
   *   fairplay: { certificateUrl, url, licenseHeaders }
   * }
   *
   * 注意：
   * - ライセンスリクエストのヘッダは、外部システム側で払い出したトークン等を注入してください。
   * - CORS/HTTPS/Content-Type は必須です（ブラウザがライセンスURLへ到達できること）。
   */
  if (!shakaPlayer || !drmConfig) {
    return;
  }

  const config = {
    drm: {
      servers: {},
      // retryParameters: {
      //   maxAttempts: 3,
      //   baseDelay: 1000,
      //   backoffFactor: 2,
      //   fuzzFactor: 0.1,
      //   timeout: 30000,
      // },
    },
    // streaming: {
    //   bufferBehind: 30,
    //   bufferingGoal: 10,
    //   rebufferingGoal: 2,
    //   bufferingGoalDefault: 10,
    //   rebufferingGoalDefault: 2,
    //   bufferBehindDefault: 30,
    // },
    // abr: {
    //   enabled: true,
    //   defaultBandwidthEstimate: 1000000,
    //   useNetworkInformation: true,
    // },
  };

  // ===== Widevine（Chrome/Android など）=====
  if (
    drmConfig.widevine &&
    drmConfig.widevine.url &&
    drmConfig.widevine.licenseHeaders
  ) {
    config.drm.servers["com.widevine.alpha"] = drmConfig.widevine.url;

    // if (drmConfig.widevine.audioContentType) {
    //   config.drm["com.widevine.alpha"] = {
    //     audioContentType: drmConfig.widevine.audioContentType,
    //     videoContentType: drmConfig.widevine_0.videoContentType,
    //   };
    // }
  }

  let req, cert;

  // ===== FairPlay（主に Safari）=====
  // FairPlay は serverCertificate が必要なケースが多く、certificateUrl から取得してセットします。
  const requestFairplay = Boolean(
    drmConfig.fairplay &&
      drmConfig.fairplay.certificateUrl &&
      drmConfig.fairplay.url &&
      drmConfig.fairplay.licenseHeaders
  );

  if (requestFairplay) {
    req = await fetch(
      drmConfig.fairplay.certificateUrl ??
        "https://lic.staging.drmtoday.com/license-server-fairplay/cert/orca"
    );
    cert = await req.arrayBuffer();

    config.drm.servers["com.apple.fps"] = drmConfig.fairplay.url;
    config.drm.advanced = {
      "com.apple.fps": {
        serverCertificate: new Uint8Array(cert),
      },
    };
  }

  // ===== PlayReady（主に Edge/Windows など）=====
  if (
    drmConfig.playready &&
    drmConfig.playready.url &&
    drmConfig.playready.licenseHeaders
  ) {
    config.drm.servers["com.microsoft.playready"] = config.drm.servers[
      "com.microsoft.playready"
    ] = drmConfig.playready.url;
  }

  // Apply configuration
  shakaPlayer.configure(config);

  shakaPlayer
    .getNetworkingEngine()
    .registerRequestFilter(function (type, request) {
      if (type == shaka.net.NetworkingEngine.RequestType.LICENSE) {
        /**
         * ライセンスリクエストにヘッダを注入します。
         * - 今は `x-dt-custom-data` だけを代表例として入れています。
         * - 必要なら `licenseHeaders` に入れたものをここで “全部” 展開する運用にしてください。
         *
         * 例）request.headers["x-dt-auth-token"] = "...";
         */
        const headers =
          drmConfig?.widevine?.licenseHeaders ??
          drmConfig?.fairplay?.licenseHeaders ??
          drmConfig?.playready?.licenseHeaders ??
          {};

        // 代表例（DRMtoday 系でよく使う）
        if (headers["x-dt-custom-data"]) {
          request.headers["x-dt-custom-data"] = headers["x-dt-custom-data"];
        }
      }
    });

  if (requestFairplay) {
    shakaPlayer
      .getNetworkingEngine()
      .registerResponseFilter((type, response) => {
        if (type === shaka.net.NetworkingEngine.RequestType.LICENSE) {
          if (shakaPlayer.keySystem() === "com.apple.fps") {
            // FairPlay ライセンス応答は base64 文字列で返ることがあり、binary に戻す必要がある
            let responseText = shaka.util.StringUtils.fromUTF8(response.data);
            response.data =
              shaka.util.Uint8ArrayUtils.fromBase64(responseText).buffer;
          }
        }
      });
  }

  // Event listener for errors
  shakaPlayer.addEventListener("error", (event) => {
    const error = event.detail;
    console.error("Shaka DRM Error:", error);

    // if (drmConfig.onError) {
    //   drmConfig.onError(error);
    // }
  });

  console.log("Shaka DRM configured successfully");
}

/**
 * Adaptive streaming
 */
export function setupAdaptiveStreaming(shakaPlayer, streamingConfig = {}) {
  const config = {
    streaming: {
      bufferBehind: 30,
      bufferingGoal: 10,
      rebufferingGoal: 2,
      ...streamingConfig,
    },
    abr: {
      enabled: true,
      defaultBandwidthEstimate: 1000000,
      useNetworkInformation: true,
      ...streamingConfig.abr,
    },
    manifest: {
      dash: {
        clockSyncUri: "",
        ignoreDrmInfo: false,
        ...streamingConfig.dash,
      },
      hls: {
        ...streamingConfig.hls,
      },
    },
  };

  shakaPlayer.configure(config);
}

/**
 * Bitrate management
 */
export function setupBitrateManagement(shakaPlayer, bitrateConfig = {}) {
  const config = {
    abr: {
      enabled: bitrateConfig.enabled !== false,
      defaultBandwidthEstimate: bitrateConfig.defaultBandwidth || 1000000,
      useNetworkInformation: bitrateConfig.useNetworkInfo !== false,
      restrictions: {
        minWidth: bitrateConfig.minWidth,
        maxWidth: bitrateConfig.maxWidth,
        minPixels: bitrateConfig.minPixels,
        maxPixels: bitrateConfig.maxPixels,
        minFrameRate: bitrateConfig.minFrameRate,
        maxFrameRate: bitrateConfig.maxFrameRate,
        minBandwidth: bitrateConfig.minBandwidth,
        maxBandwidth: bitrateConfig.maxBandwidth,
      },
    },
  };

  shakaPlayer.configure(config);
}

/**
 * Caption/subtitle
 */
export function setupTextTracks(shakaPlayer, textConfig = {}) {
  if (textConfig.preferredLanguages) {
    shakaPlayer.configure({
      preferredAudioLanguage: textConfig.preferredLanguages.audio || "en",
      preferredTextLanguage: textConfig.preferredLanguages.text || "en",
    });
  }

  if (textConfig.tryStartWithClosedCaptions) {
    shakaPlayer.configure({
      textTrackRestrictions: {
        onlyForcedSubtitles: false,
      },
    });
  }
}

export default {
  setupShakaDRM,
  setupAdaptiveStreaming,
  setupBitrateManagement,
  setupTextTracks,
};
