import { formatQualityLabel, sortQualityTracks } from '../helpers/renditionHelper';

const AUTO_QUALITY_LEVEL = {
  id: -1,
  label: 'Auto',
  height: null,
  width: null,
  bandwidth: null
};

function _getQuality(tech, shakaPlayer) { 
  /**
   * Shaka の VariantTracks（解像度/ビットレート違い）を取得して、
   * SettingsPopup 側で表示しやすい形式に整形します。
   *
   * - 先頭に “Auto” を入れておき、id=-1 を「ABR有効」の合図にしています。
   * - Shaka 側の track.id がある場合はそれを使い、無い場合は配列インデックスで代用します。
   */
  const tracks = [AUTO_QUALITY_LEVEL];

  const levels = shakaPlayer.getVariantTracks();
  console.log('Raw variant tracks:', levels);

  levels.forEach(function(level, index) {
    tracks.push({
      id: level.id ?? index,
      label: formatQualityLabel(level) || `Quality ${index + 1}`,
      height: level.height,
      width: level.width,
      bandwidth: level.bandwidth,
    });
  });

  // UI表示用にソート（Auto を先頭、残りは基本的に高解像度→低解像度の順）
  return tracks.sort(sortQualityTracks);
}

export default function setupQualityTracks(tech, shakaPlayer) {  
  /**
   * Video.js Tech 側へ `loadedqualitydata` を投げて、設定UI（SettingsPopup）へ画質候補を渡します。
   * また、UIから選ばれた id を受け取り Shaka に反映するコールバックも一緒に渡します。
   *
   * 実体としては:
   * - Auto: `abr.enabled = true`
   * - 固定: `selectVariantTrack(..., clearBuffer=true)` で明示選択 + ABR無効
   */
  const qualityData = _getQuality(tech, shakaPlayer);
  console.log('Quality tracks found:', qualityData);

  tech.trigger('loadedqualitydata', {
    qualityData: {
      video: qualityData
    },
    qualitySwitchCallback: function(id, type) {
      console.log('Quality switch callback called with id:', id, type);
      if (type !== 'video') return;

      // Auto のときは ABR（自動適応）を有効化、固定画質のときは無効化
      shakaPlayer.configure({
        abr: {
          enabled: id === AUTO_QUALITY_LEVEL.id
        }
      });

      // Auto の場合は ABR に任せるのでここで終了
      if (id === AUTO_QUALITY_LEVEL.id) return;

      // Shaka の track.id と一致する VariantTrack を探して選択
      const tracks = shakaPlayer.getVariantTracks().filter(t => Number(t.id) === Number(id));
      console.log('Selected variant track:', tracks[0]);

      if (tracks.length > 0) {
        shakaPlayer.selectVariantTrack(tracks[0], /* clearBuffer */ true);
        
        // Video.js 側にも「画質が変わった」ことを通知（UI側の表示更新などに使える）
        tech.player().trigger('qualitychange', {
          quality: tracks[0]
        });
      }

      // fire `variantchanged` event - only supports debug mode right now
      // todo - need to figure out how to do this in non debug mode
      // if (shaka.util.FakeEvent) {
      //   const event = new shaka.util.FakeEvent('variantchanged');
      //   shakaPlayer.dispatchEvent(event);
      // }
    }
  });
}