import videojs from 'video.js';

// TODO: Modify this later if needed
/**
 * Setup audio tracks. Take the tracks from dash and add the tracks to videojs. Listen for when
 * videojs changes tracks and apply that to the dash player because videojs doesn't do this
 * natively.
 *
 * @private
 * @param {videojs} tech the videojs player tech instance
 * @param {videojs.tech} tech the videojs tech being used
 */
function handleAudioTracksAdded(tech, shakaPlayer, tracks) {
  /**
   * Shaka の “音声言語トラック” を Video.js の `audioTracks()` に同期します。
   *
   * なぜ必要？
   * - Shaka は音声言語/Role の概念を持つが、Video.js の標準UIだけでは Shaka の音声切替を自動反映しません。
   * - そこで Video.js 側に AudioTrack を作り、Video.js 側の選択変更を Shaka に反映します。
   *
   * ここで作る `videojs.AudioTrack` は UI/イベント連携のための “表示用ラッパー” です。
   */

  const videojsAudioTracks = tech.audioTracks();

  function generateIdFromTrackIndex(index) {
    return `dash-audio-${index}`;
  }

  function generateLabelFromTrack(track) {
    // label は UI に出る文字列（例: "en (main)" のように role を括弧付け）
    let label = track.language;

    if (track.role) {
      label += ` (${track.role})`;
    }

    return label;
  }

  function findDashAudioTrack(subDashAudioTracks, videojsAudioTrack) {
    return subDashAudioTracks.find((track) =>
      generateLabelFromTrack(track) === videojsAudioTrack.label
    );
  }

  // Safari creates a single native `AudioTrack` (not `videojs.AudioTrack`) when loading. Clear all
  // automatically generated audio tracks so we can create them all ourself.
  if (videojsAudioTracks.length) {
    // Safari 対策：自動生成されたトラックを消して、こちらで全て作り直す
    tech.clearTracks(['audio']);
  }

  const currentAudioTrack = tracks[0];

  tracks.forEach((dashTrack, index) => {
    const label = generateLabelFromTrack(dashTrack);

    if (dashTrack === currentAudioTrack) {
      // 外部/UI 連携用のイベント（必要なら Settings UI で表示更新に使える）
      tech.trigger('shakaaudiotrackchange', {
        language: dashTrack.language
      });
    }

    // Add the track to the player's audio track list.
    videojsAudioTracks.addTrack(
      new videojs.AudioTrack({
        enabled: dashTrack === currentAudioTrack,
        id: generateIdFromTrackIndex(index),
        kind: 'main',
        label,
        language: dashTrack.language
      })
    );
  });

  const audioTracksChangeHandler = () => {
    // Video.js 側で有効化された AudioTrack を検出し、Shaka 側の音声言語へ反映する
    for (let i = 0; i < videojsAudioTracks.length; i++) {
      const track = videojsAudioTracks[i];

      if (track.enabled) {
        // Find the audio track we just selected by the id
        const dashAudioTrack = findDashAudioTrack(tracks, track);

        if (dashAudioTrack) {
          // Set is as the current track
          tech.trigger('shakaaudiotrackchange', {
            language: dashAudioTrack.language
          });
          // Shaka に音声言語（+ role）を適用
          shakaPlayer.selectAudioLanguage(dashAudioTrack.language, dashAudioTrack.role);

          // Stop looping
          continue;
        }

      }
    }
  };

  // Video.js の audioTracks change を購読し、Shaka へ同期
  videojsAudioTracks.addEventListener('change', audioTracksChangeHandler);
  shakaPlayer.addEventListener('unloading', () => {
    // ソース切替/破棄時にイベントを外す
    videojsAudioTracks.removeEventListener('change', audioTracksChangeHandler);
  });
}

export default function setupAudioTracks(tech, shakaPlayer) {
  // Shaka が持つ「音声言語 + role」の一覧を取り、Video.js の AudioTrack に反映
  handleAudioTracksAdded(tech, shakaPlayer, shakaPlayer.getAudioLanguagesAndRoles());
}
