import videojs from "video.js";
import "video.js/dist/video-js.css";
import { setupDRM } from "./drm-config";
import CustomControlBar from "./components/CustomControlBar";
import Shaka from "./shaka/shaka.js";
import "./styles/styles.css";
import {
  DISPLAY_FORMAT,
  DISPLAY_FORMAT_VALUES,
} from "./constants/displayFormat.js";
import { detectSourceType } from "./helpers/helper.js";

/**
 * VideoJS DRM Player（ブラウザ用プレイヤー）
 *
 * - Video.js をベースに、DRM（EME / DRMtoday 等）とカスタムUIをまとめた薄いラッパーです。
 * - `new VideoJSDRMPlayer({ containerId, source, ... })` で初期化すると、
 *   指定したコンテナの中に `<video>` を自動生成して Video.js を起動します。
 *
 * 【重要】このファイルはライブラリのエントリーポイントです。
 * - embed 版（`embed/`）では、このソースから webpack で JS/CSS を生成しています。
 */

// Shaka Tech の内部状態（Video.js の Tech が参照する “state” 的な領域）
Shaka.defaultState = {};

// Video.js に Shaka Tech を登録（`techOrder: ["shaka", "html5"]` で利用可能になる）
videojs.registerTech("shaka", Shaka);

class VideoJSDRMPlayer {
  constructor(options = {}) {
    /**
     * options（代表的なもの）
     * - containerId {string} 必須: プレイヤーを挿入するDOM要素の id
     * - source {string} 必須: 動画URL（HLS/DASH/MP4等）
     * - isDRM {boolean} DRMコンテンツの場合 true
     * - drmConfig {object} isDRM=true の場合必須（ライセンスURLやヘッダ等）
     * - useShakaTech {boolean} true で Shaka Tech を使う（DASH/HLS等の拡張に有利）
     * - classicControls {boolean} true で Video.js 標準コントロールを表示
     * - controls {boolean} false でコントロール自体を無効化
     * - fluid {boolean} レスポンシブ（親要素に合わせて伸縮）
     * - audioOnlyMode {boolean} 音声のみUI（ビデオ面積を縮退）にするための補助
     * - displayFormat {number} audioOnlyMode の表示形式（HIDDEN / PLAY_PAUSE_ONLY 等）
     */

    // 必須パラメータの検証
    if (!options.containerId) {
      throw new Error("containerId parameter is required");
    }

    if (!options.source) {
      throw new Error("source parameter is required");
    }

    // DRM の場合は drmConfig が必須（ライセンスURL/ヘッダがないと再生不可）
    if (
      options.isDRM &&
      (!options.drmConfig || !Object.keys(options.drmConfig).length)
    ) {
      throw new Error("drmConfig parameter is required if isDRM is true");
    }

    this.options = {
      // Default options
      totalWidth: "100%",
      totalHeight:
        options.audioOnlyMode && options.classicControls ? "" : "auto",
      isDRM: false,
      useShakaTech: false,
      displayFormat: DISPLAY_FORMAT.DEFAULT,
      audioOnlyMode: false,
      classicControls: false,
      controls: true,
      // Merge with user options
      ...options,
    };

    this.containerId = options.containerId;
    this.source = options.source;
    this.isDRM = options.isDRM ?? false;
    this.player = null;

    // 1) コンテナ内に video タグを生成
    this.createVideoElement();

    // 2) Video.js を初期化し、ソース/DRM/UI をセットアップ
    this.initializePlayer();
  }

  createVideoElement() {
    const containerElement = document.getElementById(this.containerId);
    if (!containerElement) {
      throw new Error(
        `Container element with id "${this.containerId}" not found`
      );
    }

    // コンテナ内に Video.js 用の video タグを作成
    const videoElement = document.createElement("video");
    videoElement.id = `${this.containerId}-video`;
    videoElement.className = "video-js vjs-default-skin";
    if (!this.options.classicControls) {
      // classicControls=false のとき、Video.js 標準のコントロールバーを隠す
      videoElement.classList.add("hide-videojs-controls");
    }
    videoElement.setAttribute("playsinline", true);
    videoElement.setAttribute("webkit-playsinline", true);
    videoElement.setAttribute("x-webkit-airplay", "allow");

    if (this.options.totalWidth) {
      videoElement.style.width = this.options.totalWidth;
    }
    if (this.options.totalHeight) {
      videoElement.style.height = this.options.totalHeight;
    }

    // コンテナを一度クリアして video タグを差し込む
    containerElement.innerHTML = "";
    containerElement.appendChild(videoElement);

    // 既存プレイヤーが居れば破棄（再初期化対応）
    this.dispose();

    this.videoElement = videoElement;

    if (this.options.audioOnlyMode && !this.options.classicControls) {
      // audio-only 表示（CSS側でビデオ領域を縮退させる）
      this.videoElement.classList.add("audio-only");

      if (DISPLAY_FORMAT_VALUES.includes(this.options.displayFormat)) {
        this.videoElement.classList.add(
          `display-format-${this.options.displayFormat}`
        );

        if (this.options.displayFormat === DISPLAY_FORMAT.HIDDEN) {
          // “完全非表示” モード（音声だけ再生したいなど）
          this.videoElement.style.display = "none";
        }
      } else {
        console.warn(`Unknown displayFormat: ${this.options.displayFormat}`);
      }
    }
  }

  initializePlayer() {
    const playerOptions = {
      ...this.getVideoJSOptions(),
      // useShakaTech=true のとき Shaka Tech を優先して使う
      // （Shaka 側でDRM/トラック制御をしやすい）
      techOrder: this.options.useShakaTech ? ["shaka", "html5"] : ["html5"],
    };

    this.player = videojs(this.videoElement, playerOptions);

    if (!this.player) {
      throw new Error("Failed to initialize Video.js player");
    }

    if (this.options.fluid) {
      // Video.js の fluid（レスポンシブ）モード
      this.videoElement.classList.add("vjs-fluid");
      this.player.fluid(true);
    }

    // ページ離脱時の破棄（this を維持するため arrow function を使う）
    window.addEventListener("beforeunload", () => {
      this.dispose();
    });

    // エラーを外部に通知（CustomEvent で emit）
    this.player.on("error", (error) => {
      console.error("Video.js Player Error:", error);
      this.emit("error", error);
    });

    // Player ready 後に src/DRM/UI を設定する
    this.player.ready(() => {
      console.log("Video.js Player is ready");

      // Shaka Tech を使う場合、containerId を Tech に渡してデバッグ用インスタンスを expose する
      if (this.options.useShakaTech && this.player.tech()?.setContainerId) {
        this.player.tech().setContainerId(this.containerId);
      }

      // ソース設定：
      // - useShakaTech=false かつ isDRM=true → videojs-contrib-eme 経由で EME/DRM 設定
      // - それ以外 → 通常の player.src({src, type})
      if (!this.options.useShakaTech && this.isDRM) {
        setupDRM(
          this.player,
          {
            url: this.source,
            type: detectSourceType(this.source),
          },
          this.options.drmConfig
        );
      } else {
        this.player.src({
          src: this.source,
          type: detectSourceType(this.source),
        });
      }

      // 外部へ「初期化完了」を通知
      this.emit("ready");

      // Shaka Tech の場合、少し待ってから Shaka インスタンスを window に expose する
      if (this.options.useShakaTech) {
        // Shaka の attach/load が走る前後で tech/shaka_ が揃うまでの猶予
        setTimeout(() => {
          const tech = this.player.tech();
          if (tech?.shaka_ && tech?.setContainerId) {
            tech.setContainerId(this.containerId);
            tech.exposeShakaInstances();
          }
        }, 100);
      }

      // classicControls=false のとき、独自UI（CustomControlBar）を video.js の root に append
      if (!this.options.classicControls && this.options.controls) {
        const customControlBar = new CustomControlBar(this.player);
        this.player.el().appendChild(customControlBar.el());
      }

      // メタデータ取得後に audioOnlyMode 用の追加処理（Safari対策で少し待つ）
      this.player.on("loadedmetadata", () => {
        // Safari では loadedmetadata 直後だと要素が揃っていないことがあるため delay
        setTimeout(() => {
          try {
            if (
              !this.options.classicControls &&
              this.options.audioOnlyMode &&
              this.options.controls &&
              this.options.displayFormat === DISPLAY_FORMAT.PLAY_PAUSE_ONLY &&
              this.options.poster
            ) {
              const playButton = this.player
                .el()
                .querySelector(".videojs-drm-play-button");
              // PLAY_PAUSE_ONLY 表示のとき、ポスター画像をボタン背景として見せる
              playButton.style.background = null;
              playButton.style.backgroundRepeat = "no-repeat";
              playButton.style.backgroundPosition = "center";
              playButton.style.backgroundSize = "cover";
              playButton.style.backgroundImage = `url(${this.options.poster})`;
            }
          } catch (e) {
            console.error("Error applying audio-only mode:", e);
          }
        }, 500);
      });
    });
  }

  getVideoJSOptions() {
    const videojsOptions = { ...this.options };

    // Video.js へ渡してはいけない “このラッパー専用” オプションを除去
    delete videojsOptions.containerId;
    delete videojsOptions.source;
    delete videojsOptions.isDRM;
    delete videojsOptions.totalWidth;
    delete videojsOptions.totalHeight;
    delete videojsOptions.displayFormat;
    delete videojsOptions.classicControls;
    if (this.options.useShakaTech && this.options.isDRM) {
      // Shaka Tech の場合は、Video.js のオプションとして `shaka.drmConfig` を渡す
      videojsOptions.shaka = {
        drmConfig: this.options.drmConfig,
      };
    } else {
      delete videojsOptions.drmConfig;
    }

    return videojsOptions;
  }

  // ===== Custom event system（外部システム連携用）=====
  // DOM CustomEvent で `containerId-videojs-drm-player:<event>` という名前のイベントを発火します。
  emit(eventName, data = null) {
    const event = new CustomEvent(
      `${this.containerId}-videojs-drm-player:${eventName}`,
      {
        detail: data,
      }
    );
    document.dispatchEvent(event);
  }

  // 例: instance.on("ready", () => { ... })
  on(eventName, callback) {
    document.addEventListener(
      `${this.containerId}-videojs-drm-player:${eventName}`,
      callback
    );
  }

  // ===== Public API methods（外部から操作するための薄いラッパー）=====
  play() {
    if (this.player) {
      return this.player.play();
    }
  }

  pause() {
    if (this.player) {
      this.player.pause();
    }
  }

  stop() {
    if (this.player) {
      this.player.pause();
      this.player.currentTime(0);
    }
  }

  rewind(seconds = 10) {
    if (this.player) {
      const currentTime = this.player.currentTime();
      this.player.currentTime(Math.max(0, currentTime - seconds));
    }
  }

  forward(seconds = 10) {
    if (this.player) {
      const currentTime = this.player.currentTime();
      const duration = this.player.duration();
      this.player.currentTime(Math.min(duration, currentTime + seconds));
    }
  }

  setVolume(volume) {
    if (this.player && volume >= 0 && volume <= 1) {
      this.player.volume(volume);
    }
  }

  getVolume() {
    return this.player ? this.player.volume() : 0;
  }

  isMuted() {
    return this.player ? this.player.muted() : false;
  }

  mute() {
    if (this.player) {
      this.player.muted(true);
    }
  }

  unmute() {
    if (this.player) {
      this.player.muted(false);
    }
  }

  toggleMute() {
    if (this.player) {
      this.player.muted(!this.player.muted());
    }
  }

  getCurrentTime() {
    return this.player ? this.player.currentTime() : 0;
  }

  getDuration() {
    return this.player ? this.player.duration() : 0;
  }

  seek(time) {
    if (this.player && time >= 0) {
      this.player.currentTime(time);
    }
  }

  setSize(width, height) {
    if (this.player) {
      this.player.dimensions(width, height);
      this.options.width = width;
      this.options.height = height;
    }
  }

  // 破棄（メモリリーク防止）
  dispose() {
    if (this.player && !this.player.isDisposed()) {
      this.player.dispose();
      this.player = null;
    }

    // video 要素自体はコンテナ側で管理される前提のため、ここでは参照だけ外す
    this.videoElement = null;
  }
}

// Export for different module systems（ESM/UMD の両対応）
export default VideoJSDRMPlayer;

// UMD として `<script>` 直読み込みした場合も使えるよう window にも公開
if (typeof window !== "undefined") {
  window.VideoJSDRMPlayer = VideoJSDRMPlayer;
}
