import videojs from "video.js";
import "../../../styles/settings-popup.css";
import {
  formatQualityLabel,
  sortQualityTracks,
} from "../../../helpers/renditionHelper";
import {
  playbackSpeedIcon,
  qualityIcon,
  subtitleIcon,
  chevronLeftIcon,
  checkIcon,
} from "../../../assets/icons/index.js";

const Component = videojs.getComponent("Component");

const AUTO_QUALITY_ID = -1;
const DEFAULT_QUALITY = {
  id: AUTO_QUALITY_ID,
  label: "Auto",
};

class SettingsPopup extends Component {
  constructor(player, options) {
    super(player, options);
    /**
     * 設定ポップアップ（歯車ボタンで開くメニュー）
     *
     * 目的：
     * - 再生速度（playbackRate）
     * - 画質（HLSの場合は Video.js/VHS、Shakaの場合は Shaka の VariantTrack）
     * - 字幕（Video.js textTracks / Shaka textTracks を UI で ON/OFF）
     *
     * 補足：
     * - Shaka Tech 使用時は `loadedqualitydata` イベント + qualitySwitchCallback を受け取り、
     *   Shaka の ABR/variant 選択を直接切り替えます。
     */
    this.isPopupOpen = false;
    this.currentSpeed = 1;
    this.currentRendition = DEFAULT_QUALITY;
    this.renditions = [];
    this.currentSubtitle = null;
    this.subtitles = [];
    this.currentMenu = "main"; // 'main', 'playback-speed', 'quality', or 'subtitles'
  }

  createEl() {
    const el = super.createEl("div", {
      className: "videojs-drm-settings-popup",
    });
    el.style.display = "none";

    return el;
  }

  ready() {
    super.ready();

    // Shaka Tech 判定（Shaka の場合は画質切替ロジックが異なる）
    this.isShakaPlayer =
      this.player().options().useShakaTech &&
      this.player().techName_ === "Shaka";
    // 初回表示用のメニューを組み立て
    this.createMainMenu();

    this.attachEvents();
    this.setupRenditionListener();
    this.setupSubtitleListener();
  }

  // Main popup container
  createMainMenu() {
    const mainMenuOptions = [
      {
        action: "playback-speed",
        icon: playbackSpeedIcon,
        title: "Playback speed",
        value: this.currentSpeed === 1 ? "Normal" : this.currentSpeed + "x",
      },
      {
        action: "quality",
        icon: qualityIcon,
        title: "Quality",
        value: this.currentRendition?.label,
      },
      {
        action: "subtitles",
        icon: subtitleIcon,
        title: "Subtitles",
        value: this.currentSubtitle
          ? this.currentSubtitle.label || this.currentSubtitle.language || "Unknown"
          : "Off",
      },
    ];

    this.mainMenuContent = mainMenuOptions
      .map(
        (option) => `
        <div class="vjsd-main-menu settings-popup-item" data-action="${option.action}">
            ${option.icon}
            <span class="settings-text settings-title">${option.title}</span>
            <span class="settings-value">${option.value}</span>
        </div>
    `
      )
      .join("");
  }

  createPlaybackSpeedMenu() {
    // 再生速度の候補（Video.js の playbackRate にそのまま渡す）
    const speedOptions = [
      { value: 0.5, label: "0.5x", selected: this.currentSpeed === 0.5 },
      { value: 0.75, label: "0.75x", selected: this.currentSpeed === 0.75 },
      { value: 1, label: "Normal", selected: this.currentSpeed === 1 },
      { value: 1.25, label: "1.25x", selected: this.currentSpeed === 1.25 },
      { value: 1.5, label: "1.5x", selected: this.currentSpeed === 1.5 },
      { value: 2, label: "2x", selected: this.currentSpeed === 2 },
    ];

    this.playbackSpeedMenuContent = this.createSubmenu(
      "Playback speed",
      "speed",
      speedOptions,
      ["speed-item"]
    );
  }

  createQualityMenu() {
    // Auto を先頭に（Auto=ABR有効 の意味で扱う）
    const qualityOptions = [];
    if (!this.renditions.some((r) => r.id == AUTO_QUALITY_ID)) {
      qualityOptions.push({
        value: AUTO_QUALITY_ID,
        label: "Auto",
        selected: this.currentRendition?.id == AUTO_QUALITY_ID,
      });
    }

    // 実際に選べる画質一覧（ShakaまたはVHS由来）
    this.renditions.forEach((rendition) => {
      qualityOptions.push({
        value: rendition.id,
        label: rendition.label,
        selected: this.currentRendition?.id == rendition.id,
      });
    });

    this.qualityMenuContent = this.createSubmenu(
      "Quality",
      "quality",
      qualityOptions,
      ["quality-item"]
    );
  }

  createSubtitlesMenu() {
    // 字幕OFF（デフォルト）
    const subtitleOptions = [{
        value: 'off',
        label: 'Off',
        selected: !this.currentSubtitle || this.currentSubtitle.id === 'off',
    }];

    // 利用可能な字幕トラック
    this.subtitles.forEach((subtitle) => {
      subtitleOptions.push({
        value: subtitle.id,
        label: subtitle.label || subtitle.language || "Unknown",
        selected: this.currentSubtitle?.id == subtitle.id,
      });
    });

    this.subtitlesMenuContent = this.createSubmenu(
      "Subtitles",
      "subtitle",
      subtitleOptions,
      ["subtitle-item"]
    );
  }

  createSubmenu(title, action, options, classNames) {
    return `
      ${this.createBackButton(title)}
      ${options
        .map(
          (option) => `
        <div class="settings-popup-item ${classNames.join(" ")} ${
            option.selected ? "selected" : ""
          }" data-${action}="${option.value}">
            <span class="settings-text">${option.label}</span>
            ${option.selected ? checkIcon : ""}
        </div>
        `
        )
        .join("")}
    `;
  }

  createBackButton(title) {
    return `
      <div class="settings-popup-item back-button" data-action="back">
        ${chevronLeftIcon}
        <span class="settings-text settings-title">${title}</span>
      </div>
    `;
  }

  attachEvents() {
    // 動的にメニューが差し替わるので event delegation で一箇所に集約
    const handleInteraction = (e) => {
      e.stopPropagation();

      const item = e.target.closest(".settings-popup-item");
      if (!item) return;

      const action = item.dataset.action;
      const speed = item.dataset.speed;
      const quality = item.dataset.quality;
      const subtitle = item.dataset.subtitle;

      if (action === "playback-speed") {
        this.showPlaybackSpeedMenu();
      } else if (action === "quality") {
        this.showQualityMenu();
      } else if (action === "subtitles") {
        this.showSubtitlesMenu();
      } else if (action === "back") {
        this.showMainMenu();
      } else if (speed) {
        this.setPlaybackSpeed(parseFloat(speed));
      } else if (quality) {
        this.setQuality(quality);
      } else if (subtitle) {
        this.setSubtitle(subtitle);
      }
    };

    // モバイル対応：click と touchend の両方を処理
    this.el().addEventListener("click", handleInteraction);
    this.el().addEventListener("touchend", (e) => {
      // Prevent click event from firing after touchend
      e.preventDefault();
      handleInteraction(e);
    });
  }

  toggle() {
    this.isPopupOpen = !this.isPopupOpen;
    if (this.isPopupOpen) {
      this.show();
    } else {
      this.hide();
    }
  }

  show() {
    this.el().style.display = "block";
    this.isPopupOpen = true;
    this.currentMenu = "main";

    this.createMainMenu();
    this.showMainMenu();

    setTimeout(() => {
      this.outsideHandler = (evt) => {
        // ポップアップ外をクリックしたら閉じる
        if (!this.el().contains(evt.target)) {
          this.hide();
        }
      };
      document.addEventListener("click", this.outsideHandler);
    }, 0);
  }

  hide() {
    this.el().style.display = "none";
    this.isPopupOpen = false;
    this.currentMenu = "main";

    if (this.outsideHandler) {
      document.removeEventListener("click", this.outsideHandler);
      this.outsideHandler = null;
    }
  }

  showMainMenu() {
    this.currentMenu = "main";
    this.el().innerHTML = this.mainMenuContent;
  }

  showPlaybackSpeedMenu() {
    this.currentMenu = "playback-speed";
    // Ensure content is up to date
    this.createPlaybackSpeedMenu();
    this.el().innerHTML = this.playbackSpeedMenuContent;
  }

  showQualityMenu() {
    this.currentMenu = "quality";
    // Ensure content is up to date
    this.createQualityMenu();
    this.el().innerHTML = this.qualityMenuContent;
  }

  showSubtitlesMenu() {
    this.currentMenu = "subtitles";
    // Ensure content is up to date
    this.createSubtitlesMenu();
    this.el().innerHTML = this.subtitlesMenuContent;
  }

  setPlaybackSpeed(speed) {
    this.currentSpeed = speed;
    if (this.player) {
      // Video.js API：playbackRate を変更
      this.player().playbackRate(speed);
    }
    // Close popup after selection
    this.hide();
  }

  setQuality(qualityId) {
    try {
      // UIで選んだ id を、renditions 配列から引く
      const rendition = this.renditions.find((r) => r.id == qualityId) || null;
      if (!rendition) return;

      this.currentRendition = rendition;
      if (this.isShakaPlayer && this.qualitySwitchCallback) {
        // Shaka Tech の場合：setup-quality-tracks が渡した callback で Shaka に適用
        this.qualitySwitchCallback(rendition.id, "video");
        return;
      }

      // VHS / qualityLevels 経由の場合：Video.js 側の表現（representation/qualityLevel）を切替
      this.setRendition(rendition);
    } catch (error) {
      console.warn("Could not set quality:", error);
    } finally {
      this.hide();
    }
  }

  setSubtitle(subtitleId) {
    if (subtitleId === "off") {
      this.currentSubtitle = null;
      this.disableSubtitles();
    } else {
      const subtitle = this.subtitles.find((s) => s.id == subtitleId);
      if (subtitle) {
        this.currentSubtitle = subtitle;
        this.enableSubtitle(subtitle);
      }
    }
    // Close popup after selection
    this.hide();
  }

  setupRenditionListener() {
    if (!this.player()) return;

    const tech = this.player().tech();
    if (tech && this.isShakaPlayer) {
      tech.on(
        "loadedqualitydata",
        (event, { qualityData, qualitySwitchCallback }) => {
          // Shaka から画質情報を受け取る（loadedqualitydata は setup-quality-tracks が発火）
          this.qualitySwitchCallback = qualitySwitchCallback;
          this.updateRenditionsFromShaka(qualityData);
        }
      );
    }

    this.player().on("qualitytrackchange", (event, options) => {
      // Video.js 側の quality変更通知（UI表示の追従用）
      this.updateCurrentRendition(options);
    });

    if (!this.isShakaPlayer) {
      // Listen for loadedmetadata to get renditions
      this.player().on("loadedmetadata", () => {
        this.updateRenditions();
      });

      // Listen for source changes
      this.player().on("loadstart", () => {
        this.updateRenditions();
      });

      // Listen for quality changes
      this.player().on("qualitychange", () => {
        this.updateRenditions();
      });
    }
  }

  setupSubtitleListener() {
    if (!this.player) return;
    
    // this.player().tech().on("texttracksloaded", () => {
    //   this.updateSubtitles();
    // });

    // Listen for loadedmetadata to get subtitles
    this.player().on("loadedmetadata", () => {
      // Set timeout to ensure text tracks are loaded (for Safari)
      setTimeout(() => {
        // Video.js textTracks を読み取り、字幕候補を作り直す
        this.updateSubtitles();
      }, 1000);
    });

    // Listen for subtitle changes from CCButton
    this.player().on('subtitlechanged', () => {
      this.updateCurrentSubtitle();
    });

    // Listen for source changes
    // this.player().on("loadstart", () => {
    //   this.updateSubtitles();
    // });

    // Listen for text track changes
    // this.player().on("texttrackchange", () => {
    //   this.updateCurrentSubtitle();
    // });
  }

  updateRenditionsFromShaka(qualityData) {
    try {
      if (qualityData?.video?.length > 0) {
        // Shaka の qualityData（Auto + variants）を UI 用の renditions に変換
        this.renditions = qualityData.video.map((track, index) => ({
          id: track.id,
          bandwidth: track.bandwidth,
          width: track.width,
          height: track.height,
          label: track.label,
          index: index,
          originalTrack: track,
        }));

        // Keep the order from quality data (Auto first, then by height, then by bandwidth)
        this.currentRendition =
          this.renditions.find((r) => r.id == AUTO_QUALITY_ID) || null;
      }
    } catch (error) {
      console.warn("Could not update renditions from quality data:", error);
    }
  }

  updateRenditions() {
    console.log("Updating renditions");
    try {
      // Try using qualityLevels API first
      if (
        this.player().qualityLevels &&
        this.player().qualityLevels()?.length > 0
      ) {
        // qualityLevels が使える場合（主に HLS/VHS）：level.enabled の切替で画質固定ができる
        this.renditions = Array.from(this.player().qualityLevels())
          .map((level, index) => ({
            id: `quality-${index}`,
            label: formatQualityLabel(level),
            bandwidth: level.bandwidth,
            width: level.width,
            height: level.height,
            index: index,
            qualityLevel: level,
          }))
          .sort(sortQualityTracks);

        console.log(
          "Available renditions from qualityLevels:",
          this.renditions
        );
      } else if (this.player().tech()?.vhs) {
        console.log("Quality levels is not usable, trying VHS...");
        const vhs = this.player().tech().vhs;

        if (vhs.representations && vhs.representations()) {
          // VHS の representations（各解像度/帯域）を UI 用に整形
          const rawReps = vhs.representations();

          this.renditions = rawReps
            .map((rep, index) => ({
              id: `rendition-${index}`,
              label: formatQualityLabel(rep),
              bandwidth: rep.bandwidth,
              width: rep.width,
              height: rep.height,
              resolution: rep.resolution,
              index: index,
              representation: rep,
            }))
            .sort(sortQualityTracks);

          console.log("Available renditions from VHS:", this.renditions);
        }
      }
    } catch (error) {
      console.warn("Could not get renditions:", error);
    } finally {
      // UIの都合で Auto を必ず先頭に差し込む
      if (!this.renditions.some((r) => r.id == AUTO_QUALITY_ID)) {
        this.renditions.unshift(DEFAULT_QUALITY);
      }
    }
  }

  updateCurrentRendition(options) {
    try {
      if (options && options.id !== undefined) {
        this.currentRendition =
          this.renditions.find((r) => r.id == options.id) || null;
        console.log("Current rendition updated to:", this.currentRendition);
      }
    } catch (error) {
      console.warn("Could not update current rendition:", error);
    }
  }

  setRendition(rendition = null) {
    try {
      // Auto を選んだら “全部有効” にして ABR に委ねる
      const toAutoRendition = rendition.id === AUTO_QUALITY_ID;
      // Try using qualityLevels API first
      if (
        this.player().qualityLevels &&
        this.player().qualityLevels()?.length > 0
      ) {
        // Disable all quality levels first
        Array.from(this.player().qualityLevels()).forEach((level, index) => {
          level.enabled = toAutoRendition;
        });

        // Enable only the selected quality level
        if (rendition.qualityLevel) {
          rendition.qualityLevel.enabled = true;
          console.log(
            "Enabled quality level:",
            rendition.qualityLevel ?? rendition.label
          );
        }
      } else if (this.player().tech()?.vhs && rendition.representation) {
        const vhs = this.player().tech().vhs;
        const reps = vhs.representations();

        reps.forEach((rep) => {
          if (rep === rendition.representation) {
            rep.enabled = true;
          } else {
            rep.enabled = toAutoRendition;
          }
        });
        console.log("Set rendition to:", rendition);

        // Force VHS to update
        if (vhs.trigger) {
          vhs.trigger("renditionchange");
        }
      }
    } catch (error) {
      console.warn("Could not set rendition:", error);
    }
  }

  updateSubtitles() {
    try {
      let textTracks = this.player().textTracks();

      // textTracks の中から字幕/キャプションだけを抽出して UI 用配列にする
      this.subtitles = Array.from(textTracks)
        .filter(
          (track) =>
            track.kind === "captions" ||
            track.kind === "subtitles" ||
            track.kind === "subtitle"
        )
        .map((track, index) => ({
          id: track.id || `subtitle-${index}`,
          language: track.language || "unknown",
          label: track.label || track.language || `Subtitle ${index + 1}`,
          kind: track.kind,
          index: index,
          textTrack: track,
        }));

      console.log("Available subtitles:", this.subtitles);

      // Update current subtitle based on active track
      this.updateCurrentSubtitle();
    } catch (error) {
      console.warn("Could not get subtitles:", error);
    }
  }

  updateCurrentSubtitle() {
    try {
      const textTracks = this.player().textTracks();
      const activeTrack = Array.from(textTracks).find(
        (track) =>
          track.mode === "showing" &&
          (track.kind === "captions" ||
            track.kind === "subtitles" ||
            track.kind === "subtitle")
      );

      if (activeTrack) {
        this.currentSubtitle = this.subtitles.find(
          (subtitle) => subtitle.textTrack === activeTrack
        );
      } else {
        this.currentSubtitle = null;
      }

      console.log("Current subtitle:", this.currentSubtitle);
    } catch (error) {
      console.warn("Could not update current subtitle:", error);
    }
  }

  enableSubtitle(subtitle) {
    try {
      console.log("Enabling subtitle:", subtitle);

      // Disable all text tracks first
      const textTracks = this.player().textTracks();
      Array.from(textTracks).forEach((track) => {
        track.mode = "disabled";
        if (track === subtitle.textTrack) {
          // 目的のトラックだけ showing にする
          track.mode = "showing";
        }
      });

      this.updateCCButtonState();
    } catch (error) {
      console.warn("Could not enable subtitle:", error);
    }
  }

  disableSubtitles() {
    try {
      const textTracks = this.player().textTracks();
      Array.from(textTracks).forEach((track) => {
        if (track.kind === "captions" || track.kind === "subtitles" || track.kind === "subtitle") {
          track.mode = "disabled";
        }
      });

      this.updateCCButtonState();
    } catch (error) {
      console.warn("Could not disable subtitles:", error);
    }
  }
  
  updateCCButtonState() {
    // CCButton と状態を共有するためのイベント（CCButton 側がこれを購読して見た目更新）
    this.player().trigger('subtitlechanged');
  }

  dispose() {
    super.dispose();
    this.el().remove();
  }
}

videojs.registerComponent("SettingsPopup", SettingsPopup);

export default SettingsPopup;
