import videojs from 'video.js';
import { 
  PlayButton, 
  VolumeButton, 
  VolumeSlider, 
  TimeDisplay, 
  CCButton,
  SettingsButton,
  FullscreenButton,
  ProgressBar
} from './control-bar/components';
import '../styles/custom-control-bar.css';

const Component = videojs.getComponent('Component');

class CustomControlBar extends Component {
  constructor(player) {
    super(player);

    console.log('CustomControlBar ready, setting up controls...');
    /**
     * このコンポーネントは Video.js の標準 `ControlBar` を置き換えるための “独自UI” です。
     * `VideoJSDRMPlayer` 側で classicControls=false の場合にプレイヤーDOMへ append されます。
     *
     * 構造（createEl 内のHTML）：
     * - 上段：プログレスバー（シーク）
     * - 下段左：再生/音量/時間
     * - 下段右：字幕/設定/フルスクリーン
     */
    this.setupControls();
  }

  createEl() {
    const el = super.createEl('div', {
      className: 'custom-video-control-bar'
    });

    // コントロールバーの骨組みだけを先に生成し、
    // 各ボタンコンポーネントは setupControls() でそれぞれのコンテナへ差し込みます。
    el.innerHTML = `
      <div class="custom-progress-container"></div>
      <div class="custom-controls-container">
        <div class="custom-controls-left">
          <div class="custom-play-button-container"></div>
          <div class="custom-volume-container"></div>
          <div class="custom-time-display-container"></div>
        </div>
        <div class="custom-controls-right">
          <div class="custom-cc-button-container"></div>
          <div class="custom-settings-button-container"></div>
          <div class="custom-fullscreen-button-container"></div>
        </div>
      </div>
    `;

    return el;
  }

  ready() {
    super.ready();
    console.log('CustomControlBar ready!');
  }

  setupControls() {
    // タッチ端末は hover が無いので、ボリュームスライダーの出し分けを行う
    const isTouchDevice = 'ontouchstart' in window || navigator.maxTouchPoints > 0;

    // Progress Bar（シーク）
    const progressBar = new ProgressBar(this.player(), {});
    const progressContainer = this.el().querySelector('.custom-progress-container');
    progressContainer.appendChild(progressBar.el());

    // Play Button
    const playButton = new PlayButton(this.player(), {});
    const playContainer = this.el().querySelector('.custom-play-button-container');
    playContainer.appendChild(playButton.el());

    // Volume（ボタン + スライダー）
    const volumeContainer = this.el().querySelector('.custom-volume-container');
    const volumeButton = new VolumeButton(this.player(), {});
    volumeContainer.appendChild(volumeButton.el());
    if (!isTouchDevice) {
      const volumeSlider = new VolumeSlider(this.player(), {});
      volumeContainer.appendChild(volumeSlider.el());
      this.setupVolumeSliderHover(volumeContainer, volumeSlider.el());
    }

    if (!this.player().options_.audioOnlyMode) {
      // 時間表示（audioOnlyMode のときは省略）
      const timeDisplay = new TimeDisplay(this.player(), {});
      const timeContainer = this.el().querySelector('.custom-time-display-container');
      timeContainer.appendChild(timeDisplay.el());

      // 字幕ボタン（CC）
      const ccButton = new CCButton(this.player(), {});
      const ccContainer = this.el().querySelector('.custom-cc-button-container');
      ccContainer.appendChild(ccButton.el());

      // 設定ポップアップ（再生速度/画質/字幕）
      const settingsButton = new SettingsButton(this.player(), {});
      const settingsContainer = this.el().querySelector('.custom-settings-button-container');
      settingsContainer.appendChild(settingsButton.el());

      // フルスクリーン
      const fullscreenButton = new FullscreenButton(this.player(), {
        controlBar: this.el()
      });
      const fullscreenContainer = this.el().querySelector('.custom-fullscreen-button-container');
      fullscreenContainer.appendChild(fullscreenButton.el());
    }
  }


  setupVolumeSliderHover(vContainer, vSlider) {
    if (!vContainer || !vSlider) return;

    // マウスが外れたらスライダーを隠す（タッチ端末は hover が無いので使わない）
    const hideSlider = () => {
      vSlider.style.opacity = "0";
      vSlider.style.width = "0";
      vSlider.style.paddingRight = "0";
    };

    // 初期状態は隠す
    hideSlider();

    // hover で表示
    vContainer.addEventListener("mouseenter", () => {
      this.clearVolumeTimer();
      vSlider.style.opacity = "1";
      vSlider.style.width = null;
      vSlider.style.paddingRight = "4px";
    });

    vContainer.addEventListener("mouseleave", (e) => {
      this.volumnTimer = setTimeout(() => {
        hideSlider();
      }, 200);
    });
  }

  clearVolumeTimer() {
    if (this.volumnTimer) {
      clearTimeout(this.volumnTimer);
      this.volumnTimer = null;
    }
  }

  dispose() {
    // TODO: 追加したイベントリスナの掃除をより厳密に（必要なら）
    this.clearVolumeTimer();
    super.dispose();
  }
}

// Register the component
videojs.registerComponent('CustomControlBar', CustomControlBar);

export default CustomControlBar;
